"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringType = void 0;
const tslib_1 = require("tslib");
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const type_detect_1 = tslib_1.__importDefault(require("type-detect"));
const internals_1 = require("../internals");
const type_1 = require("./type");
class StringType extends type_1.Type {
    constructor(options = {}) {
        // We want to allow empty strings, however calling `allow('')` causes
        // Joi to allow the value and skip any additional validation.
        // Instead, we reimplement the string validator manually except in the
        // hostname case where empty strings aren't allowed anyways.
        let schema = options.hostname === true
            ? internals_1.internals.string().hostname()
            : internals_1.internals.any().custom((value) => {
                if (typeof value !== 'string') {
                    return `expected value of type [string] but got [${(0, type_detect_1.default)(value)}]`;
                }
            });
        if (options.minLength !== undefined) {
            schema = schema.custom((value) => {
                if (value.length < options.minLength) {
                    return `value has length [${value.length}] but it must have a minimum length of [${options.minLength}].`;
                }
            });
        }
        if (options.maxLength !== undefined) {
            schema = schema.custom((value) => {
                if (value.length > options.maxLength) {
                    return `value has length [${value.length}] but it must have a maximum length of [${options.maxLength}].`;
                }
            });
        }
        super(schema, options);
    }
    handleError(type, { limit, value }) {
        switch (type) {
            case 'any.required':
                return `expected value of type [string] but got [${(0, type_detect_1.default)(value)}]`;
            case 'string.hostname':
                return `value must be a valid hostname (see RFC 1123).`;
        }
    }
}
exports.StringType = StringType;
