// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, } from '@aws-crypto/material-management';
export function getAwsKmsMrkAwareDiscoveryMultiKeyringBuilder(MrkAwareDiscoveryKeyring, MultiKeyring, defaultClientProvider) {
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
    //# The caller MUST provide:
    return function buildAwsKmsMrkAwareDiscoveryMultiKeyringNode({ regions, discoveryFilter, 
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
    //# If a regional client supplier is not passed,
    //# then a default MUST be created that takes a region string and
    //# generates a default AWS SDK client for the given region.
    clientProvider = defaultClientProvider, grantTokens, }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# If an empty set of Region is provided this function MUST fail.
        needs(regions.length, 'Configured regions must contain at least one region.');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# If
        //# any element of the set of regions is null or an empty string this
        //# function MUST fail.
        needs(regions.every((region) => typeof region === 'string' && !!region), 'Configured regions must not contain a null or empty string as a region.');
        const children = regions
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
            //# A set of AWS KMS clients MUST be created by calling regional client
            //# supplier for each region in the input set of regions.
            .map(clientProvider)
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
            //# Then a set of AWS KMS MRK Aware Symmetric Region Discovery Keyring
            //# (aws-kms-mrk-aware-symmetric-region-discovery-keyring.md) MUST be
            //# created for each AWS KMS client by initializing each keyring with
            .map((client) => {
            /* Postcondition: If the configured clientProvider is not able to create a client for a defined region, throw an error. */
            needs(client, 'Configured clientProvider is unable to create a client for a configured region.');
            return new MrkAwareDiscoveryKeyring({
                client,
                discoveryFilter,
                grantTokens,
            });
        });
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# Then a Multi-Keyring (../multi-keyring.md#inputs) MUST be initialize
        //# by using this set of discovery keyrings as the child keyrings
        //# (../multi-keyring.md#child-keyrings).
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-multi-keyrings.txt#2.5
        //# This Multi-Keyring MUST be
        //# this functions output.
        return new MultiKeyring({ children });
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX21ya19kaXNjb3ZlcnlfbXVsdGlfa2V5cmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9rbXNfbXJrX2Rpc2NvdmVyeV9tdWx0aV9rZXlyaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLG9FQUFvRTtBQUNwRSxzQ0FBc0M7QUFFdEMsT0FBTyxFQUNMLEtBQUssR0FJTixNQUFNLGlDQUFpQyxDQUFBO0FBdUJ4QyxNQUFNLFVBQVUsNkNBQTZDLENBSTNELHdCQUVDLEVBQ0QsWUFBc0MsRUFDdEMscUJBQWdEO0lBRWhELHlFQUF5RTtJQUN6RSw0QkFBNEI7SUFDNUIsT0FBTyxTQUFTLDRDQUE0QyxDQUFDLEVBQzNELE9BQU8sRUFDUCxlQUFlO0lBQ2YseUVBQXlFO0lBQ3pFLGdEQUFnRDtJQUNoRCxpRUFBaUU7SUFDakUsNERBQTREO0lBQzVELGNBQWMsR0FBRyxxQkFBcUIsRUFDdEMsV0FBVyxHQUNzQztRQUNqRCx5RUFBeUU7UUFDekUsa0VBQWtFO1FBQ2xFLEtBQUssQ0FDSCxPQUFPLENBQUMsTUFBTSxFQUNkLHNEQUFzRCxDQUN2RCxDQUFBO1FBRUQseUVBQXlFO1FBQ3pFLE1BQU07UUFDTixxRUFBcUU7UUFDckUsdUJBQXVCO1FBQ3ZCLEtBQUssQ0FDSCxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQyxPQUFPLE1BQU0sS0FBSyxRQUFRLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQyxFQUNqRSx5RUFBeUUsQ0FDMUUsQ0FBQTtRQUVELE1BQU0sUUFBUSxHQUNaLE9BQU87WUFDTCx5RUFBeUU7WUFDekUsdUVBQXVFO1lBQ3ZFLHlEQUF5RDthQUN4RCxHQUFHLENBQUMsY0FBYyxDQUFDO1lBQ3BCLHlFQUF5RTtZQUN6RSxzRUFBc0U7WUFDdEUscUVBQXFFO1lBQ3JFLHFFQUFxRTthQUNwRSxHQUFHLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtZQUNkLDBIQUEwSDtZQUMxSCxLQUFLLENBQ0gsTUFBTSxFQUNOLGlGQUFpRixDQUNsRixDQUFBO1lBQ0QsT0FBTyxJQUFJLHdCQUF3QixDQUFDO2dCQUNsQyxNQUFNO2dCQUNOLGVBQWU7Z0JBQ2YsV0FBVzthQUNaLENBQUMsQ0FBQTtRQUNKLENBQUMsQ0FBQyxDQUFBO1FBRU4seUVBQXlFO1FBQ3pFLHdFQUF3RTtRQUN4RSxpRUFBaUU7UUFDakUseUNBQXlDO1FBQ3pDLEVBQUU7UUFDRix5RUFBeUU7UUFDekUsOEJBQThCO1FBQzlCLDBCQUEwQjtRQUMxQixPQUFPLElBQUksWUFBWSxDQUFDLEVBQUUsUUFBUSxFQUFFLENBQUMsQ0FBQTtJQUN2QyxDQUFDLENBQUE7QUFDSCxDQUFDIn0=